#
# asDataView_Vxy
#   (c) ASkr, 2011
#   www.askrprojects.net
#
#---------------------------------------------------------------------------------------
# WARNING #1:
#
#    CODE USES TABs. BEST VIEWED WITH TAB == 2
#
#---------------------------------------------------------------------------------------
# WARNING #2:
#
#    SCROLL DOWN TO THE PARAMETER SECTION AND ADJUST SETTINGS!
#
#
#---------------------------------------------------------------------------------------
# CHANGES
#
#   V0.9b:
#     - added TAB-SHIFT shortcut (decrease current channel number)
#     - added key-repeat function
#     - added all-other-waveforms-off key 'o'
#     - added all-waveforms-on key 'a'
#     
#
#
#
#

#---------------------------------------------------------------------------------------
#
# A minimalistic and simple, but quite useful app for saving debug data via
# an UART connection. Just a quick hack...
#
# The main purpose of this little, stinky app was saving debug data (sensors or other
# measurement data) to disk.
# But sometimes it is extremly helpful to view any data before saving it (overload
# conditions, HW gain adjustments, etc...), so I quickly hacked in a real-time
# data display...
#
# Notice that Python has a speed limit.
# This app is for sure not able to display 100kS/s 16 bit data for 23 channels ;)
#
# E.g., I used it for:
#  - acquiring 1/f noise measurement data (2 channels, 12 bit, 3072 S/s)
#  - PSoC capacitive sensor data display (10 channels, 8-12 bit, 20-100 S/s)
#  - a lot more ;)
#
# Theoretically, there's no limit on the number of channels (except for Python/Computer
# performance or serial transmission time).
#
# Notice that there is no menu or any in-app settings screen.
# You'll need to adjust the parameters in the PARAMETER section below...
#
#---------------------------------------------------------------------------------------
# REQUIREMENTS:
#
#  - Python       http://www.python.org  
#  - PySerial     http://pyserial.sourceforge.net
#  - PyGame       http://pygame.org
#
#
#---------------------------------------------------------------------------------------
# CREDITS:
#
#  Rotating menu from Francesco Mastellone. Great!
#  http://pygame.org/project-Rotating+Menu-975-.html
#
#
#---------------------------------------------------------------------------------------
# NOTES:
#
#  - If your computer is too slow and can't capture all data, try one of the following:
#     - turn off waveforms by hitting 'w'
#     - decrease display time to 1s (hit cursor 'RIGHT' until you see 't=1')
#     - decrease window size (faster drawing)
#     - increase display update skips by hitting '.' (period)
#       Now, data is drawn only during every n-th main loop.
#    The mouse pointer inside the application window is a good performance indicator.
#    If movement is choppy (less than 2-3 updates per second), action must be taken.
#
#  - For a real-time display, hit 't' (and wait for at least pDAT_SRATEUPDATETIME seconds).
#    The app tries to approximate the sampling time and will adjust the display to fit
#    the selected time (cursor 'LEFT' or 'RIGHT').
#
#---------------------------------------------------------------------------------------
# UART PACKET DESCRIPTION:
#
#   +------+------+------+------+------+------+------+------+------+------+
#   | 0x10 | 0x02 |  B1  |  B2  |  B3  |  B4  |  Bn  | 0x10 | 0x02 |  B1  |      
#   +------+------+------+------+------+------+------+------+------+------+
#   |<-- START -->|  CH1   ...                       |<-- START -->|
#
#   Every packet starts with a 2 byte header (0x10, 0x02), followed by an
#   arbitrary length of user data.
#   Channel resolution can be anything in between 1..16 bits, hence transmitting
#   a single channel requires 1 byte for resolutions of 1..8 bits, or 2 bytes for
#   a resolution for 9..16 bits.
#   In case of the latter, the low byte is transmitted first:
#
#     data to send = 0x1234
#     +------+------+------+------+...
#     | 0x10 | 0x02 | 0x34 | 0x12 |...
#     +------+------+------+------+...
#     |<-- START -->|<--- CH1 --->|...
#
#   Every data byte containing a 0x10 needs to be repeated a second time.
#
#
#     Example, 3 x 8 bit channels:
#       CH1 data = 0xAA
#       CH2 data = 0xBB
#       CH3 data = 0xCC
#     +------+------+------+------+------+------+------+-...
#     | 0x10 | 0x02 | 0xAA | 0xBB | 0xCC | 0x10 | 0x02 | ...
#     +------+------+------+------+------+------+------+-...
#     |<-- START -->|  CH1 |  CH2 |  CH3 |<-- START -->| ...
#
#
#     Example, 3 x 8 bit channels:
#       CH1 data = 0xAA
#       CH2 data = 0x10
#       CH3 data = 0xCC
#     +------+------+------+------+------+------+------+------+-...
#     | 0x10 | 0x02 | 0xAA | 0x10 | 0x10 | 0xCC | 0x10 | 0x02 | ...
#     +------+------+------+------+------+------+------+------+-...
#     |<-- START -->|  CH1 |     CH2     |  CH3 |<-- START -->| ...
#
#
#     Example, 1 x 16 and 2 x 8 bit channels:
#       CH1 data = 0xAABB
#       CH2 data = 0xCC
#       CH3 data = 0xDD
#     +------+------+------+------+------+------+------+------+-...
#     | 0x10 | 0x02 | 0xBB | 0xAA | 0xCC | 0xDD | 0x10 | 0x02 | ...
#     +------+------+------+------+------+------+------+------+-...
#     |<-- START -->|     CH1     |  CH2 |  CH3 |<-- START -->| ...
#
#
#     Example, 1 x 16 and 2 x 8 bit channels ("worst case 0x10 scenario"):
#       CH1 data = 0x1010
#       CH2 data = 0x10
#       CH3 data = 0x10
#     +------+------+------+------+------+------+------+------+------+------+------+------+-...
#     | 0x10 | 0x02 | 0x10 | 0x10 | 0x10 | 0x10 | 0x10 | 0x10 | 0x10 | 0x10 | 0x10 | 0x02 | ...
#     +------+------+------+------+------+------+------+------+------+------+------+------+-...
#     |<-- START -->|            CH1            |     CH2     |     CH3     |<-- START -->| ...
#
#
#---------------------------------------------------------------------------------------
# MENU:
#
#  MENU/START  -  open serial interface and start sampling
#  MENU/SAVE   -  save previously recorded waveforms to disk (one file per channel)
#  MENU/QUIT   -  quit
#
#---------------------------------------------------------------------------------------
# HOTKEYS (during acquisistion):
#
#  ESC   -  close serial interface and stop sampling
#  TAB   -  channel selection upwards, hold left CTRL key to reverse direction
#  r     -  reset data (delete all previously acquired samples)
#  w     -  toggle selected waveform visibility
#  o     -  turn on selected waveform and hide all other
#  a     -  turn on all waveforms
#  .     -  increase display update skips by 100 (data gets drawn only every n loop)
#  ,     -  decrease display update skips by 100 (data gets drawn only every n loop)
#  i     -  toggle info message
#  t     -  real-time sampling rate update and display
#  u     -  toggle sampling rate display unit (samples/s <-> time/sample); 
#  v     -  toggle value display at cursor and marker (units depend on pDAT_UREFS)
#  MB1/3 -  mouse buttons set value markers on current channel (best used after stopped with "SPACE")
#  SPACE -  toggle (stop/start) waveform and background updates update
#  UP    -  increase waveform display offset
#  DOWN  -  decrease waveform display offset
#  LEFT  -  decrease display time by 1s
#  RIGHT -  increase display time by 1s
#  0     -  set waveform display scale factor to 1
#  8     -  decrease waveform display scale factor by 1
#  9     -  increase waveform display scale factor by 1 (can be tricky in conjunction with offset)
#
#
#---------------------------------------------------------------------------------------
# PARAMETERS:
#
pSER_COMPORT = 6									# serial comm port number (COM1 = 1, COM2 = 2, ...)
pDAT_BAUDRATE = 1000000						# baud rate in bits/s
pSER_INBUF = 8192									# serial buffer (8192 is a nice, default value even for "higher" baud rates)
pDAT_SRATE = 100									# initial sampling rate guess; use key 't' to update during acquisition
pDAT_SRATEUPDATETIME = 3					# update time interval of sampling rate (if enabled via key 't')
pDAT_CHANNELS = 4									# number of channels in use

# If you need more channels, just increase the length of the three following lists.
pDAT_RESOLS = [	# resolution of data channel; valid: 2**1 ... 2**16
								2**12,						# CH1
								2**12,						# CH2
								2**16,						# CH3
								2**16,						# CH4
								2**16,						# CH5
								2**8,							# CH6
								2**8,							# CH7
								2**8,							# CH8
								2**16,						# CH9
								2**8,							# CH10
								2**8,							# CH11
								2**8,							# CH12, extend, if necessary...
							]
pDAT_UREFS  = [ # reference value (use '1' for RAW values)
								3.3,							# CH1
								3.3,							# CH2
								1,								# CH3
								1,								# CH4
								1,								# CH5
								3.3,							# CH6
								5.0,							# CH7
								5.0,							# CH8
								1,								# CH5
								3.3,							# CH6
								1,								# CH11
								1,								# CH12, extend, if necessary...
							]
pDAT_COLORS = [ # channel colors
								(000,000,000),		# CH1
								(000,255,000),		# CH2
								(000,000,255),		# CH3
								(255,255,255),		# CH4
								(000,100,255),		# CH5
								(255,000,255),		# CH6
								(255,000,000),		# CH7
								(000,000,000),		# CH8
								(000,100,255),		# CH9
								(255,000,255),		# CH10
								(255,000,000),		# CH11
								(000,000,000),		# CH12, extend, if necessary...
							]


							
#---------------------------------------------------------------------------------------
# NO CHANGES BELOW HERE
pVER_STRING = "asDataView_V09"
pWIN_SIZE = 640, 320												# initial window size
pWIN_TIME = 10															# initial time in s to show
pWIN_BGCOLORIDLE = (130, 130, 130)
pWIN_BGCOLORSAMP = (250, 150, 100) 					# you might wish to change this ;)
pWIN_BGCOLOROVER = (250,   0,   0)
pMEN_SPEED = 50

import pygame
import sys
import random
import serial
import time

from pygame.locals import *
from string import maketrans
from math import sin, cos, pi



########################################################################################
### CLASS SerAcq
###
########################################################################################
class SerAcq:

	def __init__(self):
		self.port = None
		self.serial = None

	def Open(self, port):
		self.port = port
		if self.serial == None:
			try:
				self.serial = serial.Serial(port-1, pDAT_BAUDRATE, timeout=0, parity=serial.PARITY_NONE)
				return 1
			except:
				print("unable to open serial port " + str(self.port))
				self.serial = None
				return 0
		else:
			return 2
			
	def Close(self):
		if self.serial != None:
			self.serial.close()
			self.serial = None
			return 1
		else:
			return 0

	def Read(self, length):
		if self.serial != None:
			return self.serial.read(length)
		else:
			return ''

	def Flush(self):
		self.serial.flushInput()
		

########################################################################################
### CLASS Data
###
### Class for data storage (including some additional display paramenters).
### WARNING: no type checks!
########################################################################################
class Data:

	def __init__(self, limit, resol, uref, offset = 0, scale = 1, color = (0,0,0), hidden = False ):
		self.limit = int(limit)
		self.resol = resol
		self.uref = uref
		self.offset = offset;
		if scale < 1:
			scale = 1
		self.scale = scale
		self.color = color
		self.hidden = hidden
		self.values = [None] * self.limit

	def Add(self, values):
		self.values += values
		if len(self.values) > self.limit:
			self.values = self.values[len(self.values)-self.limit:]

	def Reset(self):
		self.values = [None] * self.limit
	
	def Get(self):
		return self.values

	def SetLimit(self, limit):
		if limit < self.limit:
			self.values = self.values[-limit:]
		elif limit > self.limit:
			for i in range(limit - self.limit):
				self.values.insert(0,None)
		self.limit = limit
	
	# adds (or subtract) a value to the offset parameter
	def Offset(self, offset = 0):
		self.offset += offset
		return self.offset
	
	# add (or subtract) a value to the scale parameter
	def Scale(self, scale = 0):
		if scale == None:
			self.scale = 1
		else:
			self.scale += scale
		if self.scale < 1:
			self.scale = 1
		return self.scale

	def SetColor(self, color):
		self.color = color

	def GetColor(self):
		return self.color

	def SetHidden(self, hidden = None):
		if hidden == True:
			self.hidden = True
		elif hidden == False:
			self.hidden = False
		else:
			if self.hidden == True:
				self.hidden = False
			else:
				self.hidden = True

	def GetHidden(self):
		return self.hidden
	
	def GetResol(self):
		return self.resol
		
	def GetUref(self):
		return self.uref

		
########################################################################################
### CLASS DataWindow
###   WARNING: no error checks!
########################################################################################
class DataWindow:

	def __init__(self):
		self.data = {}
		self.time = pWIN_TIME
		self.size = pWIN_SIZE
		self.srate = pDAT_SRATE
		self.bgcolor = pWIN_BGCOLORIDLE
		self.screen = pygame.display.set_mode(self.size, RESIZABLE)
		self.surface = pygame.Surface(self.screen.get_size()).convert()
		self.font = pygame.font.Font(None, 24)
		self.DrawBG()
		self.Show()

	def GetSurface(self):
		return self.surface
		
	def SetBGColor(self, color):
		if isinstance(color, tuple):
			self.bgcolor = color
		else:
			self.bgcolor = pWIN_BGCOLORIDLE
		self.DrawBG()
		
	def GetSize(self):
		return self.screen.get_size()
		
	def Resize(self, newsize):
		self.size = newsize
		self.screen = pygame.display.set_mode(self.size, RESIZABLE)
		self.surface = pygame.Surface(self.screen.get_size()).convert()
		self.DrawBG()
		for n in self.data:
			self.DrawData(n)
		self.Show()

	def SetSrate(self, srate):
		self.srate = srate

	def GetSrate(self):
		return self.srate
		
	def SetTime(self,time):
		self.time += time
		if self.time < 1:
			self.time = 1
		elif self.time > 20:
			self.time = 20
		for d in self.data:
			self.data[d].SetLimit(self.time * self.srate)
		return self.time

	def Show(self):
		self.screen.blit(self.surface,(0,0))
		pygame.display.flip()

	def SetScale(self, dataobj, scale):
		return self.data[dataobj].Scale(scale)

	def SetOffset(self, dataobj, offset):
		return self.data[dataobj].Offset(offset)
		
	def DrawBG(self):
		self.surface.fill(self.bgcolor)

	# TESTING
	def DrawAny(self, obj):
		self.surface.blit(obj,(0,0))

	def DrawText(self, message, ypos = 10, color = (0,0,0) ):
		text = self.font.render(message, False, self.bgcolor, color)	
		textRect = text.get_rect()
		textRect.centerx = self.screen.get_rect().centerx
		textRect.centery = ypos
		self.surface.blit(text, textRect)
		
	def DrawTextNow(self, message, wait = 2000):
		text = self.font.render(message, False, (0,0,0), (255,0,0))	
		textRect = text.get_rect()
		textRect.centerx = self.screen.get_rect().centerx
		textRect.centery = self.screen.get_rect().centery
		self.screen.blit(text, textRect)
		pygame.display.flip()
		pygame.time.wait(wait)
		
	def DrawLines(self, lines, width = 1, color = (0,0,0) ):
		pygame.draw.lines(self.surface, color, False, lines, width)

	def DrawData(self, name):
		blockscale = float(self.size[0]) / (self.srate * self.time)
		lines = []
		loffset = self.data[name].Offset()
		lscale  = self.data[name].Scale()
		lresol  = self.data[name].GetResol()-1
		lsize   = self.size[1]
		i = 0
		for dat in self.data[name].Get():
			if dat != None:
#				j = int((dat+loffset)/float(lresol)*self.size[1]*lscale-(self.size[1]/2*(lscale-1)) )
				j = int((dat+loffset)/float(lresol)*lsize*lscale-(lsize/2*(lscale-1)) )
				lines.append((  int(i*blockscale),lsize-j-1))
			i += 1
		if len(lines) > 1:
			self.DrawLines(lines, width = 2, color = self.data[name].GetColor())

	def GetValueByYPos(self, name, ypos):
		loffset = self.data[name].Offset()
		lscale  = self.data[name].Scale()
		lresol  = self.data[name].GetResol()-1
		lsize   = self.size[1]
		val = (-1)*(((1 + ypos - (lsize*(1 + ((lscale - 1)/2.0))))*lresol/float(lsize*lscale)) + loffset)
		tmp = self.data[name].GetUref()
		if tmp == 1:
			return int(val)
		else:
			return val*tmp/lresol
			
	# create a new data row
	def DataCreate(self, name, resol, uref, color = (0,0,0) ):
		self.data[name] = Data(self.srate*self.time, resol, uref, color = color)
		return name

	# delete a data row
	def DataDelete(self, name):
		del self.data[name]
		
	# add data to a data row
	def DataAdd(self, name, values):
		self.data[name].Add(values)

	# reset a data row
	def DataReset(self, name):
		self.data[name].Reset()
		
	# get data row color
	def DataGetColor(self, name):
		return self.data[name].GetColor()
		
	# set hidden flag
	def DataSetHidden(self, name, hidden = None):
		return self.data[name].SetHidden(hidden)
	
	# get hidden flag
	def DataGetHidden(self, name):
		return self.data[name].GetHidden()



########################################################################################
### sinInterpolation
###
########################################################################################
def sinInterpolation(start, end, steps=30):
	values = [start]
	delta = end - start
	for i in range(1, steps):
		n = (pi / 2.0) * (i / float(steps - 1))
		values.append(start + delta * sin(n))
	return values

		
########################################################################################
### RotatingMenu
###
########################################################################################
class RotatingMenu:
		def __init__(self, x, y, radius, arc=pi*2, defaultAngle=0, wrap=False):
				"""
				@param x:
						The horizontal center of this menu in pixels.
				
				@param y:
						The vertical center of this menu in pixels.
				
				@param radius:
						The radius of this menu in pixels(note that this is the size of
						the circular path in which the elements are placed, the actual
						size of the menu may vary depending on item sizes.
				@param arc:
						The arc in radians which the menu covers. pi*2 is a full circle.
				
				@param defaultAngle:
						The angle at which the selected item is found.
				
				@param wrap:
						Whether the menu should select the first item after the last one
						or stop.
				"""
				self.x = x
				self.y = y
				self.radius = radius
				self.arc = arc
				self.defaultAngle = defaultAngle
				self.wrap = wrap
				
				self.rotation = 0
				self.rotationTarget = 0
				self.rotationSteps = [] #Used for interpolation
				
				self.items = []
				self.selectedItem = None
				self.selectedItemNumber = 0
				
		# added by ASkr
		def SetSize(self, x, y, r):
			self.x = x
			self.y = y
			self.radius = r

		def addItem(self, item):
				self.items.append(item)
				if len(self.items) == 1:
						self.selectedItem = item
		
		def selectItem(self, itemNumber):
				if self.wrap == True:
						if itemNumber > len(self.items) - 1: itemNumber = 0
						if itemNumber < 0: itemNumber = len(self.items) - 1
				else:
						itemNumber = min(itemNumber, len(self.items) - 1)
						itemNumber = max(itemNumber, 0)
				
				self.selectedItem.deselect()
				self.selectedItem = self.items[itemNumber]
				self.selectedItem.select()
				
				self.selectedItemNumber = itemNumber
				
				self.rotationTarget = - self.arc * (itemNumber / float(len(self.items) - 1))
				
				self.rotationSteps = sinInterpolation(self.rotation,
																							self.rotationTarget, 45)
		
		def rotate(self, angle):
				"""@param angle: The angle in radians by which the menu is rotated.
				"""
				for i in range(len(self.items)):
						item = self.items[i]
						n = i / float(len(self.items) - 1)
						rot = self.defaultAngle + angle + self.arc * n
						
						item.x = self.x + cos(rot) * self.radius
						item.y = self.y + sin(rot) * self.radius
		
		def update(self):
				if len(self.rotationSteps) > 0:
						self.rotation = self.rotationSteps.pop(0)
						self.rotate(self.rotation)
		
		def draw(self, display):
				"""@param display: A pyGame display object
				"""
				for item in self.items:
						item.draw(display)

########################################################################################
### MenuItem
###
########################################################################################
class MenuItem:
		def __init__(self, text="Spam"):
				self.text = text
				
				self.defaultColor = (0,0,0)
				self.selectedColor = (255,0,0)
				self.color = self.defaultColor
				
				self.x = 0
				self.y = 0 #The menu will edit these
				
				self.font = pygame.font.Font(None, 20)
				self.image = self.font.render(self.text, True, self.color)
				size = self.font.size(self.text)
				self.xOffset = size[0] / 2
				self.yOffset = size[1] / 2
		
		def select(self):
				"""Just visual stuff"""
				self.color = self.selectedColor
				self.redrawText()
		
		def deselect(self):
				"""Just visual stuff"""
				self.color = self.defaultColor
				self.redrawText()
		
		def redrawText(self):
				self.font = pygame.font.Font(None, 20)
				self.image = self.font.render(self.text, True, self.color)
				size = self.font.size(self.text)
				self.xOffset = size[0] / 2
				self.yOffset = size[1] / 2
		
		def draw(self, display):
				display.blit(self.image, (self.x-self.xOffset, self.y-self.yOffset))



		
		
########################################################################################
### String2List()
###
########################################################################################
def String2List(str):
	ret = []
	for i in str:
		ret += [ord(i)]
	return ret
		


########################################################################################
### ExtractData2x16()
###
########################################################################################
def ExtractData2x16(data):
	ret = [[],[]]
	while len(data) > 10:
		if data[0] == 0x10 and data[1] == 0x02:
			tmp = []
			for i in range(4):
				d = data.pop(2)
				if d == 0x10:
					d = data.pop(2)
				tmp += [d]
			data.pop(0)
			ret[0] += [256*tmp[1]+tmp[0]]
			ret[1] += [256*tmp[3]+tmp[2]]
		data.pop(0)
	return ret

	
########################################################################################
### ExtractData4x8()
###
########################################################################################
def ExtractData4x08(data):
	ret = [[],[],[],[]]
	while len(data) > 10:
		if data[0] == 0x10 and data[1] == 0x02:
			tmp = []
			for i in range(4):
				d = data.pop(2)
				if d == 0x10:
					d = data.pop(2)
				ret[i] += [d]
			data.pop(0)
		data.pop(0)
	return ret
	
	
########################################################################################
### ExtractData()
###
########################################################################################
def ExtractData(data, emarker):
	le = len(emarker)
	ret = []
	for i in range(1,le+1):
		ret.append([])

	lmin = sum(emarker)*2+2					# total, save minimum number of bytes to read
		
	while len(data) > lmin:
		if data[0] == 0x10 and data[1] == 0x02:
			for i in range(le):
				d = data.pop(2)
				if d == 0x10:
					d = data.pop(2)
				if emarker[i] > 1:
					d2 = data.pop(2)
					if d2 == 0x10:
						d2 = data.pop(2)
					d += 256 * d2
				ret[i].append(d)
			data.pop(0)
		data.pop(0)
	return ret

	
########################################################################################
### SaveData()
###
########################################################################################
def SaveData(fname, ldata, resol, uref):
	
	try:
		fout = open(fname,"w+t")
	except:
		return 0

	for i in ldata:
		if uref == 1:
			ostr = str( int(i) ) + "\n"
		else:
			ostr = str( float(i)*float(uref)/float(resol) ) + "\n"
		fout.write(ostr)

	fout.close()
	
	return 1



	
########################################################################################
### main()
###
########################################################################################
def main():

	pygame.init();
	dw = DataWindow()

	dr = []							# list of data row names
	ldata = []					# external data row values
	emarker = []				# used to determine how many bytes per data row need to be read
	for i in range(1,pDAT_CHANNELS+1):
		dr.append(dw.DataCreate(str(i), pDAT_RESOLS[i-1], pDAT_UREFS[i-1], pDAT_COLORS[i-1] ))
		ldata.append([])
		if pDAT_RESOLS[i-1] <= 256:
			emarker += [1]		# extract 1 byte per value
		else:
			emarker += [2]		# extract 2 bytes per value

	pygame.display.set_caption(pVER_STRING)
	ser = SerAcq()
	
	pygame.key.set_repeat(500,50)
	
	infoCH = 'CH1; '
	infoT  = 't=10; '
	infoO  = 'o=0; '
	infoS  = 's=1; '
	infoN  = 'n='
	infoD  = ''
	infoSR = ''
	infoC  = ''
	showinfo = 1
	actchannel = '1'
	rawdata = []
	
	srateupdate = 0
	srlastlen = 0
	srtrigtim = 0
	srnewsrate = pDAT_SRATE
	srunit = 0		# unit for sampling rate display (0 == number of samples/s, 1 == time per sample)

	infoM = ''
	cursorshow = 0
	cursorvalue = 0
	cursorm1val = 0
	cursorm2val = 0
	cursormchannel = '1'

	drawupdateinit = 0
	drawupdate = 0
	
	sampling = 0
	stopped = 0
	running = 1

	try:
		simg = pygame.image.load('asDataView.gif')
	except:
		simg = None


	menu = RotatingMenu(pWIN_SIZE[0]/2, pWIN_SIZE[1]/2, pWIN_SIZE[1]/3, arc=pi, defaultAngle=pi/2.0)
	
	menu.addItem(MenuItem("Start"))
	menu.addItem(MenuItem("Save"))
	menu.addItem(MenuItem("QUIT"))
	menu.addItem(MenuItem(""))
	menu.addItem(MenuItem(""))
	menu.addItem(MenuItem(""))
	menu.addItem(MenuItem(""))
	menu.addItem(MenuItem(""))
	menu.selectItem(0)
	MENUSTART = 0
	MENUSAVE  = 1
	MENUQUIT  = 2
	menumax = 3

	clock = pygame.time.Clock()
		
	#-------------------------------------------------------------------------------------
	while running:

		if sampling:
		
			dstr = ser.Read(pSER_INBUF)
			if len(dstr) > 0:
				rawdata += String2List(dstr)
				
			tmp = ExtractData(rawdata,emarker)

			if len(tmp[0]) > 0:
				for i in range(len(ldata)):
					dw.DataAdd(str(i+1),tmp[i])
					ldata[i] += tmp[i]
					
			if srateupdate:
				if time.clock() > srtrigtim:
					i = len(ldata[0])
					srnewsrate = (i-srlastlen) / float(pDAT_SRATEUPDATETIME )
					srlastlen = i
					if int(srnewsrate) > 0:
						if srunit == 0:
							infoSR = 'sr = ' + str(int(srnewsrate)) + '; '
						else:
							infoSR = 'sr = %2.6f' % (1.0/srnewsrate)
							infoSR += 's; '
						dw.SetSrate(int(srnewsrate))
						dw.SetTime(0)
					srtrigtim += pDAT_SRATEUPDATETIME

			if drawupdate == 0:
				drawupdate = drawupdateinit

				if not stopped:
					dw.DrawBG()
					for i in dr:
						if not dw.DataGetHidden(i):
							dw.DrawData(i)
						
				if showinfo:
					if cursorshow:
						infoC = 'y=' + str(dw.GetValueByYPos(actchannel, pygame.mouse.get_pos()[1]))
						infoM = 'M1=' + str(cursorm1val) + '; M2=' +str(cursorm2val) + '; D=' +str(cursorm1val-cursorm2val)
						dw.DrawText(infoM, ypos = 30, color = dw.DataGetColor(cursormchannel) )	
					infoN = 'n=' + str(len(ldata[0])) + '; '
					dw.DrawText(infoCH + infoT + infoO + infoS + infoD + infoN + infoSR + infoC, ypos = 10, color = dw.DataGetColor(actchannel) )	
				dw.Show()

			drawupdate -= 1
			if drawupdate < 0:
				drawupdate = 0

		# not sampling
		else:
			dw.DrawBG()
			if simg != None:
				dw.DrawAny(simg)
			menu.update()
			menu.draw(dw.GetSurface())
#			pygame.display.flip() #Show the updated scene
			dw.Show()
			clock.tick(pMEN_SPEED)
			
				
		#-----------------------------------------------------------------------------------
		for ev in pygame.event.get():

			if ev.type == QUIT:
				running = 0

			if ev.type == VIDEORESIZE:
				dw.Resize(ev.size)
				menu.SetSize(pWIN_SIZE[0]/2,pWIN_SIZE[1]/2,pWIN_SIZE[1]/3)

			if ev.type == MOUSEBUTTONDOWN:
				#---
				if ev.button == 1:
					cursormchannel = actchannel
					cursorm1val = dw.GetValueByYPos(actchannel, pygame.mouse.get_pos()[1])
				#---
				elif ev.button == 3:
					cursormchannel = actchannel
					cursorm2val = dw.GetValueByYPos(actchannel, pygame.mouse.get_pos()[1])
				
			if ev.type == KEYDOWN:

				#-------------------------------------------------------------------------------
				# KEYS, IDLE STATE
				if sampling == 0:
					#---
					if ev.key == K_RETURN and menu.selectedItemNumber == MENUQUIT:
						running = 0
						break
					#---
					if ev.key == K_RETURN and menu.selectedItemNumber == MENUSTART:
						res = ser.Open(pSER_COMPORT)
						if res == 0:
							dw.DrawTextNow('Unable to open serial port!')
						elif res == 1:
							sampling = 1
							stopped = 0
							rawdata = []
							for i in dr:
								dw.DataReset(i)
							dw.SetBGColor(pWIN_BGCOLORSAMP)
							dw.Show()
							ser.Flush()
					#---
					elif ev.key == pygame.K_LEFT:
						if menu.selectedItemNumber < menumax-1:
							menu.selectItem(menu.selectedItemNumber + 1)
					#---
					elif ev.key == pygame.K_RIGHT:
						menu.selectItem(menu.selectedItemNumber - 1)
					
					#---
					elif ev.key == K_RETURN and menu.selectedItemNumber == MENUSAVE:
						dw.SetBGColor(0)
						dw.Show()
						if len(ldata[0]) == 0:
							dw.DrawTextNow('There is nothing to save...', wait = 1000)
						else:
							for i in range(len(ldata)):
								ret = 0
								fname = 'ch'
								if i < 9:
									fname += '0'
								fname += str(i+1) + '.dat'
								ret = SaveData(fname, ldata[i], pDAT_RESOLS[i], pDAT_UREFS[i])
								if ret:
									dw.DrawTextNow(' Waveform CH' + str(i+1) + ' saved to ' + fname, wait = 250)
								else:
									dw.DrawTextNow('   Error saving waveform CH' + str(i+1), wait = 250)
					
				#-------------------------------------------------------------------------------
				# KEYS, SAMPLING
				else:
					#---
					if ev.key == K_ESCAPE:
						res = ser.Close()
						# if res == 1:
							# dw.SetBGColor(0)
							# sampling = 0
						dw.SetBGColor(0)
						sampling = 0
					#---
					elif ev.key == K_SPACE:
						if stopped == 0:
							stopped = 1
						else:
							stopped = 0
					#---
					elif ev.key == K_i:
						if showinfo:
							showinfo = 0
						else:
							showinfo = 1
						dw.DrawBG()
					#---
					elif ev.key == K_w:
						dw.DataSetHidden(actchannel)
						dw.DrawBG()
					#---
					elif ev.key == K_o:
						for i in range(pDAT_CHANNELS):
							if str(i+1) != actchannel:
								dw.DataSetHidden(str(i+1),True)
							else:
								dw.DataSetHidden(str(i+1),False)
						dw.DrawBG()
					#---
					elif ev.key == K_a:
						for i in range(pDAT_CHANNELS):
							dw.DataSetHidden(str(i+1),False)
						dw.DrawBG()
					#---
					elif ev.key == K_r:
						for i in range(len(ldata)):
							ldata[i] = []
							dw.DataReset(str(i+1))
					#---
					elif ev.key == K_PERIOD:
						drawupdateinit += 100
						infoD = 'du=' + str(drawupdateinit) + '; '
					#---
					elif ev.key == K_COMMA:
						drawupdateinit -= 100
						if drawupdateinit < 0:
							drawupdateinit = 0
							infoD = ''
						else:
							infoD = 'du=' + str(drawupdateinit) + '; '
					#---
					elif ev.key == K_t:
						if srateupdate == 0:
							srlastlen = len(ldata[0])
							srateupdate = 1
							srtrigtim = time.clock() + pDAT_SRATEUPDATETIME
							infoSR = 'sr = ...; '
						else:
							srateupdate = 0
							infoSR = ''
					#---
					elif ev.key == K_v:
						if cursorshow == 0:
							cursorshow = 1
						else:
							cursorshow = 0
							infoC = ''
					#---
					elif ev.key == K_u:
						if srunit == 0:
							srunit = 1
						else:
							srunit = 0
						infoSR = 'sr = ...; '
					#---
					elif ev.key == K_TAB and pygame.key.get_mods() & KMOD_LCTRL:
						i = int(actchannel) - 1
						if i < 1:
							i = pDAT_CHANNELS
						actchannel = str(i)
						infoCH = 'CH' + str(i) + '; '
						infoS = 's=' + str(dw.SetScale(actchannel, 0)) + '; '
						infoO = 'o=' + str(dw.SetOffset(actchannel,0)) + '; '
					#---
					elif ev.key == K_TAB:
						i = int(actchannel) + 1
						if i > pDAT_CHANNELS:
							i = 1
						actchannel = str(i)
						infoCH = 'CH' + str(i) + '; '
						infoS = 's=' + str(dw.SetScale(actchannel, 0)) + '; '
						infoO = 'o=' + str(dw.SetOffset(actchannel,0)) + '; '
					#---
					elif ev.key == K_8:
						infoS = 's=' + str(dw.SetScale(actchannel,-1)) + '; '
					#---
					elif ev.key == K_9:
						infoS = 's=' + str(dw.SetScale(actchannel, 1)) + '; '
					#---
					elif ev.key == K_0:
						infoS = 's=' + str(dw.SetScale(actchannel, None)) + '; '
						
					#---
					elif ev.key == K_UP:
						i = int(pDAT_RESOLS[int(actchannel)-1] / 40)
						if i < 1:
							i = 1
						infoO = 'o=' + str(dw.SetOffset(actchannel, i)) + '; '
					#---
					elif ev.key == K_DOWN:
						i = int(pDAT_RESOLS[int(actchannel)-1] / 40)
						if i < 1:
							i = 1
						infoO = 'o=' + str(dw.SetOffset(actchannel, -i)) + '; '
					#---
					elif ev.key == K_LEFT:
						infoT = 't=' + str(dw.SetTime(1)) + '; '
					#---
					elif ev.key == K_RIGHT:
						infoT = 't=' + str(dw.SetTime(-1)) + '; '
					
	ser.Close()
	pygame.quit()


if __name__ == '__main__':
	main()

